# styled-tools 💅

[![NPM version](https://img.shields.io/npm/v/styled-tools.svg?style=flat-square)](https://npmjs.org/package/styled-tools)
[![NPM downloads](https://img.shields.io/npm/dm/styled-tools.svg?style=flat-square)](https://npmjs.org/package/styled-tools)
[![Dependencies](https://img.shields.io/david/diegohaz/styled-tools.svg?style=flat-square)](https://david-dm.org/diegohaz/styled-tools)
[![Build Status](https://img.shields.io/travis/diegohaz/styled-tools/master.svg?style=flat-square)](https://travis-ci.org/diegohaz/styled-tools) [![Coverage Status](https://img.shields.io/codecov/c/github/diegohaz/styled-tools/master.svg?style=flat-square)](https://codecov.io/gh/diegohaz/styled-tools/branch/master)

Useful interpolated functions for [styled-components](https://github.com/styled-components/styled-components) 💅, [emotion](https://github.com/emotion-js/emotion) 👩‍🎤, [JSS](https://github.com/cssinjs/jss) and other CSS-in-JS libraries.

## Install

npm:

    npm i styled-tools

Yarn:

    yarn add styled-tools

## Usage

```jsx
import styled, { css } from "styled-components";
import { prop, ifProp, switchProp } from "styled-tools";

const Button = styled.button`
  color: ${prop("color", "red")};
  font-size: ${ifProp({ size: "large" }, "20px", "14px")};
  background-color: ${switchProp("theme", {
    dark: "blue", 
    darker: "mediumblue", 
    darkest: "darkblue" 
  })};
`;

// renders with color: blue
<Button color="blue" />

// renders with color: red
<Button />

// renders with font-size: 20px
<Button size="large" />

// renders with background-color: mediumblue
<Button theme="darker" />
```

A more complex example:

```jsx
const Button = styled.button`
  color: ${prop("theme.colors.white", "#fff")};
  font-size: ${ifProp({ size: "large" }, prop("theme.sizes.lg", "20px"), prop("theme.sizes.md", "14px"))};
  background-color: ${prop("theme.colors.black", "#000")};
  
  ${switchProp("kind", {
    dark: css`
      background-color: ${prop("theme.colors.blue", "blue")};
      border: 1px solid ${prop("theme.colors.blue", "blue")};
    `,
    darker: css`
      background-color: ${prop("theme.colors.mediumblue", "mediumblue")};
      border: 1px solid ${prop("theme.colors.mediumblue", "mediumblue")};
    `,
    darkest: css`
      background-color: ${prop("theme.colors.darkblue", "darkblue")};
      border: 1px solid ${prop("theme.colors.darkblue", "darkblue")};
    `
  })}
  
  ${ifProp("disabled", css`
    background-color: ${prop("theme.colors.gray", "#999")};
    border-color: ${prop("theme.colors.gray", "#999")};
    pointer-events: none;
  `)}
`;
```

## API

<!-- Generated by documentation.js. Update this documentation by updating the source code. -->

#### Table of Contents

-   [prop](#prop)
    -   [Parameters](#parameters)
    -   [Examples](#examples)
-   [theme](#theme)
    -   [Parameters](#parameters-1)
    -   [Examples](#examples-1)
-   [palette](#palette)
    -   [Parameters](#parameters-2)
    -   [Examples](#examples-2)
-   [ifProp](#ifprop)
    -   [Parameters](#parameters-3)
    -   [Examples](#examples-3)
-   [ifNotProp](#ifnotprop)
    -   [Parameters](#parameters-4)
    -   [Examples](#examples-4)
-   [withProp](#withprop)
    -   [Parameters](#parameters-5)
    -   [Examples](#examples-5)
-   [switchProp](#switchprop)
    -   [Parameters](#parameters-6)
    -   [Examples](#examples-6)
-   [Types](#types)
    -   [Needle](#needle)

### prop

Returns the value of `props[path]` or `defaultValue`

#### Parameters

-   `path` **[string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String)** 
-   `defaultValue` **any** 

#### Examples

```javascript
import styled from "styled-components";
import { prop } from "styled-tools";

const Button = styled.button`
  color: ${prop("color", "red")};
`;
```

Returns **PropsFn** 

### theme

Same as `prop`, except that it returns `props.theme[path]` instead of
`props[path]`.

#### Parameters

-   `path` **[string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String)** 
-   `defaultValue` **any** 

#### Examples

```javascript
import styled from "styled-components";
import { theme } from "styled-tools";

const Button = styled.button`
 color: ${theme("button.color", "red")};
`;
```

### palette

Returns `props.theme.palette[key || props.palette][tone || props.tone || 0]` or `defaultValue`.

#### Parameters

-   `keyOrTone` **([string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String) \| [number](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Number))** 
-   `toneOrDefaultValue` **any** 
-   `defaultValue` **any** 

#### Examples

```javascript
import styled, { ThemeProvider } from "styled-components";
import { palette } from "styled-tools";

const theme = {
  palette: {
    primary: ['#1976d2', '#2196f3', '#71bcf7', '#c2e2fb'],
    secondary: ['#c2185b', '#e91e63', '#f06292', '#f8bbd0']
  }
};

const Button = styled.button`
  color: ${palette(1)};                    // props.theme.palette[props.palette][1]
  color: ${palette("primary", 1)};         // props.theme.palette.primary[1]
  color: ${palette("primary")};            // props.theme.palette.primary[props.tone || 0]
  color: ${palette("primary", -1)};        // props.theme.palette.primary[3]
  color: ${palette("primary", 10)};        // props.theme.palette.primary[3]
  color: ${palette("primary", -10)};       // props.theme.palette.primary[0]
  color: ${palette("primary", 0, "red")};  // props.theme.palette.primary[0] || red
`;

<ThemeProvider theme={theme}>
  <Button palette="secondary" />
</ThemeProvider>
```

### ifProp

Returns `pass` if prop is truthy. Otherwise returns `fail`

#### Parameters

-   `test` **([Needle](#needle) \| [Array](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Array)&lt;[Needle](#needle)> | [Object](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Object))** 
-   `pass` **any**  (optional, default `""`)
-   `fail` **any**  (optional, default `""`)

#### Examples

```javascript
import styled from "styled-components";
import { ifProp, palette } from "styled-tools";

const Button = styled.button`
  background-color: ${ifProp("transparent", "transparent", palette(0))};
  color: ${ifProp(["transparent", "accent"], palette("secondary"))};
  font-size: ${ifProp({ size: "large" }, "20px", ifProp({ size: "medium" }, "16px", "12px"))};
`;
```

Returns **PropsFn** 

### ifNotProp

Returns `pass` if prop is falsy. Otherwise returns `fail`

#### Parameters

-   `test` **([Needle](#needle) \| [Array](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Array)&lt;[Needle](#needle)> | [Object](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Object))** 
-   `pass` **any** 
-   `fail` **any** 

#### Examples

```javascript
import styled from "styled-components";
import { ifNotProp } from "styled-tools";

const Button = styled.button`
  font-size: ${ifNotProp("large", "20px", "30px")};
`;
```

Returns **PropsFn** 

### withProp

Calls a function passing properties values as arguments.

#### Parameters

-   `needle` **([Needle](#needle) \| [Array](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Array)&lt;[Needle](#needle)>)** 
-   `fn` **[Function](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Statements/function)** 

#### Examples

```javascript
// example with polished
import styled from "styled-components";
import { darken } from "polished";
import { withProp, prop } from "styled-tools";

const Button = styled.button`
  border-color: ${withProp(prop("theme.primaryColor", "blue"), darken(0.5))};
  font-size: ${withProp("theme.size", size => `${size + 1}px`)};
  background: ${withProp(["foo", "bar"], (foo, bar) => `${foo}${bar}`)};
`;
```

Returns **PropsFn** 

### switchProp

Switches on a given prop. Returns the value or function for a given prop value. Third parameter is default value.

#### Parameters

-   `needle` **[Needle](#needle)** 
-   `cases` **([Object](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Object) | PropsFn)** 
-   `defaultCase` **any** 

#### Examples

```javascript
import styled, { css } from "styled-components";
import { switchProp, prop } from "styled-tools";

const Button = styled.button`
  font-size: ${switchProp(prop("size", "medium"), {
    small: prop("theme.sizes.sm", "12px"),
    medium: prop("theme.sizes.md", "16px"),
    large: prop("theme.sizes.lg", "20px")
  }, prop("theme.sizes.md", "16px"))};
  ${switchProp("theme.kind", {
    light: css`
      color: LightBlue;
    `,
    dark: css`
      color: DarkBlue;
    `
  }, css`color: black;`)}
`;

<Button size="large" theme={{ kind: "light" }} />
```

Returns **PropsFn** 

### Types




#### Needle

A Needle is used to map the props to a value. This can either be done with
a path string `"theme.size.sm"` or with a function
`(props) => props.theme.size.sm` (these two examples are equivalent).

All of styled-tools can be used as Needles making it possible to do
composition between functions. ie `ifProp(theme("dark"), "black", "white")`

Type: ([string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String) \| [Function](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Statements/function))

## License

MIT © [Diego Haz](https://github.com/diegohaz)
